#include <stdio.h>

/* All plugin sources should start including "gcc-plugin.h".  */
#include "gcc-plugin.h"
/* This let us inspect the GENERIC intermediate representation.  */
#include "tree.h"

/* All plugins must export this symbol so that they can be linked with
   GCC license-wise.  */
int plugin_is_GPL_compatible;

/* Given an enumeration type (ENUMERAL_TYPE node) and a name for it
   (IDENTIFIER_NODE), describe its enumerators on the standard
   output.  */
static void
dump_enum_type (tree enum_type, tree enum_name)
{
  printf ("Found enum %s:\n", IDENTIFIER_POINTER (enum_name));

  /* Process all enumerators.  These are encoded as a linked list of
     TREE_LIST nodes starting from TYPE_VALUES and following
     TREE_CHAIN links.  */
  for (tree v = TYPE_VALUES (enum_type);
       v != NULL;
       v = TREE_CHAIN (v))
  {
    /* Get this enumerator's value (TREE_VALUE).  Give up if it's not
       a small integer.  */
    char buffer[128] = "\"<big integer>\"";
    if (tree_fits_shwi_p (TREE_VALUE (v)))
      {
	long l = tree_to_shwi (TREE_VALUE (v));
	snprintf (buffer, 128, "%li", l);
      }

    printf ("  %s = %s\n",
	    IDENTIFIER_POINTER (TREE_PURPOSE (v)),
	    buffer);
  }
}

/* Thanks to register_callback, GCC will call the following for each
   parsed type specifier, providing the corresponding GENERIC node as
   the "gcc_data" argument.  */
static void
handle_finish_type (void *gcc_data, void *user_data)
{
  (void) user_data;
  tree t = (tree) gcc_data;

  /* Skip everything that is not a named enumeration type.  */
  if (TREE_CODE (t) != ENUMERAL_TYPE
      || TYPE_NAME (t) == NULL)
    return;

  dump_enum_type (t, TYPE_NAME (t));
}

/* Like handle_finish_type, but called instead for each parsed
   declaration.  */
static void
handle_finish_decl (void *gcc_data, void *user_data)
{
  (void) user_data;
  tree t = (tree) gcc_data;
  tree type = TREE_TYPE (t);

  /* Skip everything that is not an enumeration typedef.  */
  if (TREE_CODE (t) != TYPE_DECL
      || TREE_CODE (type) != ENUMERAL_TYPE)
    return;

  dump_enum_type (type, DECL_NAME (t));
}

/* Most interesting part so far: this is the plugin entry point.  */
int
plugin_init (struct plugin_name_args *plugin_info,
	     struct plugin_gcc_version *version)
{
  (void) version;

  /* Give GCC a proper name and version number for this plugin.  */
  const char *plugin_name = plugin_info->base_name;
  struct plugin_info pi = { "0.1", "Enum binder plugin" };
  register_callback (plugin_name, PLUGIN_INFO, NULL, &pi);

  /* Execute a callback on all parsed type specifiers.  */
  register_callback (plugin_name, PLUGIN_FINISH_TYPE,
		     &handle_finish_type, NULL);
  /* Execute a callback on all typedefs.  */
  register_callback (plugin_name, PLUGIN_FINISH_DECL,
		     &handle_finish_decl, NULL);

  return 0;
}
